<?php
header("Content-Type: application/json");

// 1️⃣ Only allow POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405); // Method Not Allowed
    echo json_encode(["status" => false, "message" => "Only POST method is allowed"]);
    exit;
}

// 2️⃣ Secret key authentication
$headers = getallheaders();
if (!isset($headers['X-API-KEY']) || $headers['X-API-KEY'] !== "111-111-111-111") {
    http_response_code(401); // Unauthorized
    echo json_encode(["status" => false, "message" => "Unauthorized"]);
    exit;
}

// 3️⃣ Get JSON request body
$input = file_get_contents("php://input");
$data = json_decode($input, true);

// If JSON invalid
if (json_last_error() !== JSON_ERROR_NONE) {
    http_response_code(400); // Bad Request
    echo json_encode(["status" => false, "message" => "Invalid JSON"]);
    exit;
}

// OTP is required
if (empty($data['otp'])) {
    http_response_code(400); // Bad Request
    echo json_encode(["status" => false, "message" => "OTP is required"]);
    exit;
}

require __DIR__ . '/../vendor/autoload.php';
use ZATCA\EGS;

// Example: take branch/device info from request
// $egs_unit = [
//     'uuid' => $data['uuid'] ?? 'default-uuid',
//     'custom_id' => $data['custom_id'] ?? 'EGS1-0000',
//     'model' => $data['model'] ?? 'IOS',
//     'CRN_number' => $data['CRN_number'] ?? '',
//     'VAT_name' => $data['VAT_name'] ?? '',
//     'VAT_number' => $data['VAT_number'] ?? '',
//     'location' => $data['location'] ?? [],
//     'branch_name' => $data['branch_name'] ?? '',
//     'branch_industry' => $data['branch_industry'] ?? '',
// ];

$egs_unit = $data['egs_unit'];
try {
    $egs = new EGS($egs_unit);
    $egs->production = false; // Sandbox mode
    // 1️⃣ Generate keys and CSR
    list($private_key, $csr) = $egs->generateNewKeysAndCSR($data['VAT_name']);
    echo json_encode([
        "status" => false,
        "message" => "Internal server error",
        "csr" => $csr ,
        "private_key" => $private_key ,
    ]);

    // 2️⃣ Request Compliance Certificate (using OTP from request)
    list($request_id, $binary_security_token, $secret) = $egs->issueComplianceCertificate($data['otp'], $csr);

    // ✅ Return success with received request
    echo json_encode([
        "status" => true,
        "message" => "Device registered successfully",
        "request" => $data, // echo back received request
        "request_id" => $request_id,
        "binary_security_token" => $binary_security_token,
        "secret" => $secret,
    ]);
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        "status" => false,
        "message" => "Internal server error",
        "error" => $e->getMessage(),
    ]);
}
